/**
 * Copyright  Sergey Melnik (Stanford University, Database Group) 
 *
 * Distribution policies are governed by the W3C software license.
 * http://www.w3.org/Consortium/Legal/copyright-software   
 * 
 * All Rights Reserved.
 *
 * @author      Sergey Melnik <melnik@db.stanford.edu>
 */

package org.w3c.rdf.util.xml;

import org.xml.sax.*;

/**
 * An implementation of the <code>ErrorHandler</code>
 */

public class ErrorStore implements ErrorHandler {

  private String		m_sErrorMsg = new String ();
  private String		m_sWarningMsg = new String ();
  private int                   m_lineNum = -1;
  private int                   m_columnNum = -1;

  public ErrorStore() {
  }

    String getExceptionMsg (SAXParseException exception)
    {
        m_lineNum = exception.getLineNumber();
        m_columnNum = exception.getColumnNumber();

	return exception.getMessage() +
	    " (" +
	    "line " +
	    exception.getLineNumber() +
	    ", column " +
	    exception.getColumnNumber() +
	    ")";
    }

    /**
     * Report all warnings, and continue parsing.
     *
     * @see org.xml.sax.ErrorHandler#warning
     */
    public void warning (SAXParseException exception)
    {
	m_sWarningMsg += getExceptionMsg(exception);
    }


    /**
     * Report all recoverable errors, and try to continue parsing.
     *
     * @see org.xml.sax.ErrorHandler#error
     */
    public void error (SAXParseException exception)
    {
	m_sErrorMsg += "Recoverable Error: " + getExceptionMsg(exception);
    }


    /**
     * Report all fatal errors, and try to continue parsing.
     *
     * <p>Note: results are no longer reliable once a fatal error has
     * been reported.</p>
     *
     * @see org.xml.sax.ErrorHandler#fatalError
     */
    public void fatalError (SAXParseException exception) throws SAXException
    {
      m_lineNum = exception.getLineNumber();
      m_columnNum = exception.getColumnNumber();
      m_sErrorMsg = exception.getMessage();

      throw new SAXException("Fatal Error: " + getExceptionMsg(exception), exception);
    }

    public String errors () {
	return m_sErrorMsg;
    }

    public String warnings () {
	return m_sWarningMsg;
    }

    public int getLineNumber () {
        return m_lineNum;
    }

    public int getColumnNumber () {
        return m_columnNum;
    }

    public String getErrorMessage () {
        return m_sErrorMsg;
    }
}

