/**
 * A simple Element class for storing the element name, attributes
 * and children.
 *
 * @author Janne Saarela
 */
package org.w3c.rdf.implementation.syntax.sirpac;

import java.net.URL;
import java.util.*;
import java.io.*;
import org.xml.sax.AttributeList;
import org.xml.sax.SAXException;

public class Element
{
    private String	m_sName = null;
    private Hashtable	m_attributes = new Hashtable();
    private Vector	m_children = new Vector();
    private String	m_sResource = null;
    private String	m_sID = null;
    private String	m_sBagID = null;
    private String	m_sAbout = null;
    private String	m_sAboutEach = null;
    private String	m_sAboutEachPrefix = null;
    private Vector	m_vTargets = new Vector ();
    private boolean	m_bDone = false;
    private String	m_sPrefix = null;

    public Element (String sName, AttributeList al) /*throws SAXException*/ {
	m_sName = sName;

	if (al != null) {
	    int iLength = al.getLength ();
	    for (int x = 0; x < iLength; x++) {
	        String aName = al.getName (x);
	        String aValue = al.getValue (x);

	        m_attributes.put (aName, aValue);
	    }
	}
    }

    public String name() {
	return m_sName;
    }

    public void prefix (String sPrefix) {
	m_sPrefix = sPrefix;
    }

    public String prefix () {
	if (m_sPrefix != null)
	    return m_sPrefix + ":";
	else
	    return "";
    }

    public void addChild (Element e) {
	m_children.addElement (e);
    }

    public Enumeration children () {
	return m_children.elements();
    }

    public Enumeration attributes () {
	return m_attributes.keys();
    }

    public void addAttribute (String sName, String sValue) {
	if (sName == null ||
	    sValue == null)
	    return;
	m_attributes.put (sName, sValue);
    }

    public void removeAttribute (String sName) {
	m_attributes.remove (sName);
    }

    public String getAttribute (String sName) {
	return (String)m_attributes.get (sName);
    }

    public void addTarget (Element e) {
	m_vTargets.addElement (e);
    }

    public Enumeration targets () {
	return m_vTargets.elements();
    }

    public Element target () {
	if (m_vTargets.size() == 0)
	    return null;
	else
	    return (Element)m_vTargets.elementAt(0);
    }

    public void resource (String sResource) {
	m_sResource = sResource;
    }

    public void resource (String sResource, String sContext) {
	m_sResource = makeAbsolute (sResource, sContext);
    }

    public String resource () {
	return m_sResource;
    }

    public void ID (String sID) {
	m_sID = sID;
    }

    public void ID (String sID, String sContext) {
	m_sID = makeAbsolute (sID, sContext);
    }

    public String ID () {
	return m_sID;
    }

    public void bagID (String sBagID) {
	m_sBagID = sBagID;
    }

    public void bagID (String sBagID, String sContext) {
	m_sBagID = makeAbsolute (sBagID, sContext);
    }

    public String bagID () {
	return m_sBagID;
    }

    public void about (String sAbout) {
	m_sAbout = sAbout;
    }

    public void about (String sAbout, String sContext) {
	m_sAbout = makeAbsolute (sAbout, sContext);
    }

    public String about () {
	return m_sAbout;
    }

    public void aboutEach (String sAboutEach) {
	m_sAboutEach = sAboutEach;
    }

    public void aboutEach (String sAboutEach, String sContext) {
	m_sAboutEach = makeAbsolute (sAboutEach, sContext);
    }

    public String aboutEach () {
	return m_sAboutEach;
    }

    public void aboutEachPrefix (String sAboutEachPrefix) {
	m_sAboutEachPrefix = sAboutEachPrefix;
    }

    public void aboutEachPrefix (String sAboutEachPrefix, String sContext) {
	m_sAboutEachPrefix = makeAbsolute (sAboutEachPrefix, sContext);
    }

    public String aboutEachPrefix () {
	return m_sAboutEachPrefix;
    }

    public void linearize (int indent, PrintStream ps) {
	for (int x = 0; x < indent; x++) {
	    ps.print (" ");
	}
	System.out.print ("Element "+name()+" (");

	Enumeration eKeys = m_attributes.keys ();
	while (eKeys.hasMoreElements()) {
	    String sName = (String)eKeys.nextElement ();
	    String sValue = (String)m_attributes.get (sName);
	    System.out.print (" "+sName+"="+sValue);
	}
	System.out.print (")\n");

	Enumeration e = children();
	while (e.hasMoreElements()) {
	    Element ele = (Element)e.nextElement();
	    ele.linearize (indent + 2, ps);
	}
    }

    public boolean done () {
	return m_bDone;
    }

    public void done (boolean b) {
	m_bDone = b;
    }

    /**
     * Private methods for this class only
     */

    public static String makeAbsolute (String sURI, String context) {

      if(sURI != null && sURI.indexOf(':') > 0)
	return sURI; // already absolute

	String sResult = new String ();
	if (sURI != null &&
	    context != null) {

	    /**
	     * If sURI has .. then it indicates relative URI which
	     * we must make absolute
	     */
	    if (sURI.startsWith ("..")) {
		try {
		    URL absoluteURL = new URL (new URL(context), sURI);
		    sResult = absoluteURL.toString();
		} catch(Exception e) {
		    System.err.println("RDF Resource - cannot combine " + 
				       context + " with " +sURI);
		}
	    } else {
		/**
		 * If sURI is an absolute URI, don't bother
		 * with it
		 */
		try {
		    URL absoluteURL = new URL (sURI);
		    sResult = sURI; // absoluteURL.toString();
		    //		    sResult = absoluteURL.toString();
		    //		    System.out.println("ABS1: " + sResult + " from " + sURI + ", " + context);

		} catch(Exception e) {
		    /**
		     * Well, the sURI wasn't absolute either,
		     * try making absolute with the context
		     */

		  // Cases: "dir/foo.rdf#" + "Top/Arts" should not produce dir/Top/Arts
		  //        "ex.rdf#prg1" + "#query2" should produce "ex.rdf#query2"
		  //		    if (sURI.indexOf ('/') > -1) {
		    if (sURI.startsWith ("#")) {
			try {
			    URL absoluteURL = new URL (new URL(context), sURI);
			    sResult = absoluteURL.toString();
			} catch (Exception e2) {
			  sResult = context +  sURI; // ???
			}
		    } else
		      sResult = context + /*"#" +*/ sURI; // ???
		    
		  //System.out.println("ABS2: " + sResult + " from " + sURI + ", " + context + ": " + e.getMessage());
		}
	    }
	    return sResult;
	} else {
	  //System.out.println("ABS3: " + sURI + " from " + sURI + ", " + context);
	    return sURI;
	}
    }

  // SM
  public String toString() {
    String res = "[Element " + name() + "(";
    for(Enumeration en = attributes(); en.hasMoreElements();) {
      String aName = (String)en.nextElement();
      String aValue = getAttribute(aName);
      res += aName + "=" + aValue + " ";
    }
    res += ")<";
    Enumeration en = children();
    while(en.hasMoreElements()) {
      res += en.nextElement().toString();
      if(en.hasMoreElements())
	res += ",";
    }
    res += ">]";
    return res;
  }

  public static void main(String[] args) throws Exception {

    Element e = new Element("", null);
    System.err.println( e.makeAbsolute(args[0], args[1]) );
  }
}
